/*******************************************************************************
********************************************************************************
**
**  Copyright(c) 2022, Alliance for Automotive Innovation
**  Used only under license from the Alliance for Automotive Innovation. All Rights Reserved.
**
**  Project:  J1699-5
**  FileName: VerifyReservedServices.c
**  Author:   EnGenius
**  Date:     2/25/2022
**  Email:    <support@autosinnovate.org>
**
**  Purpose:  SAE J1699-5 Vehicle OBD II Compliance Test Cases Source Code.
**            This source code is intended to run the tests described in
**            the SAE J1699-5 document in an automated manner, when compiled
**            and used with an SAE J2534-compatible pass-thru device.
**
**            File j1699.c contains information on building and running this test.
**
**  Description:
**
**  Modifications:  03/13/2023  Initial Version
**
********************************************************************************
*******************************************************************************/

#include <stdio.h>    // C Library input and output declarations
#include <stdlib.h>   // C Library general function declarations
#include <time.h>     // C Library time and date declarations
#include <windows.h>  // Windows API declarations
#include "j2534.h"    // j1699 project j2534 declarations
#include "j1699.h"    // j1699 project general declarations


/*******************************************************************************
**
**  Function:  VerifyReservedServices
**
**  Purpose:   Verify that all ECUs respond correctly to reserved/unused services
**
*******************************************************************************/
STATUS VerifyReservedServices ( void )
{
	REQ_MSG  stReqMsg;

	BOOL     bTestFailed = FALSE;
	BYTE     SIDIdx;


	// For each SID
	for ( SIDIdx = 0x00;
	      SIDIdx <= 0x0F;
	      SIDIdx++ )
	{
		switch ( SIDIdx )
		{
			case 0x00:
			case 0x0B:
			case 0x0C:
			case 0x0D:
			case 0x0E:
			case 0x0F:
			{
				stReqMsg.SID = SIDIdx;
				stReqMsg.NumIds = 0;
				if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL|REQ_MSG_IGNORE_NO_RESPONSE ) != FAIL )
				{
					Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
					      "SID $%02X response\n",
					      SIDIdx );
					bTestFailed = TRUE;
				}
			}
			break;
		}
	}

	if ( VerifyECUCommunication ( ) != PASS )
	{
		bTestFailed = TRUE;
	}

	if ( bTestFailed == TRUE )
	{
		return FAIL;
	}

	return PASS;
}


/*
*******************************************************************************
**  VerifyDeprecatedServices :  Check for a response to a deprecated classic OBD Service request ($1-$A)
*******************************************************************************
*/
STATUS VerifyDeprecatedServices ( unsigned int ID )
{
	REQ_MSG  stReqMsg;
	STATUS   RetCode = PASS;

	// Check if J1979 Classic Services respond
	stReqMsg.SID = ID;
	if ( ID == 0x0A || ID == 0x07 || ID == 0x03 )
	{
		stReqMsg.NumIds  = 0;
	}
	else
	{
		stReqMsg.NumIds  = 1;
		stReqMsg.u.ID[0] = 0x00;
	}

	if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL|REQ_MSG_IGNORE_NO_RESPONSE ) != FAIL )
	{
		RetCode = FAIL;
	}

	return RetCode;
}
