/*******************************************************************************
********************************************************************************
**
**  Copyright(c) 2022, Alliance for Automotive Innovation
**  Used only under license from the Alliance for Automotive Innovation. All Rights Reserved.
**
**  Project:  J1699-5
**  FileName: Test5_NoDTC.c
**  Author:   EnGenius
**  Date:     2/25/2022
**  Email:    <support@autosinnovate.org>
**
**  Purpose:  SAE J1699-5 Vehicle OBD II Compliance Test Cases Source Code.
**            This source code is intended to run the tests described in
**            the SAE J1699-5 document in an automated manner, when compiled
**            and used with an SAE J2534-compatible pass-thru device.
**
**            File j1699.c contains information on building and running this test.
**
**  Description:
**
**  Modifications:  03/13/2023  Initial Version
**
********************************************************************************
*******************************************************************************/

#include <stdio.h>    // C Library input and output declarations
#include <stdlib.h>   // C Library general function declarations
#include <time.h>     // C Library time and date declarations
#include <windows.h>  // Windows API declarations
#include "j2534.h"    // j1699 project j2534 declarations
#include "j1699.h"    // j1699 project general declarations


// Function Prototypes
extern STATUS VerifyAllINF10Data ( void );
extern STATUS CheckMIL ( void );
extern STATUS VerifyReverseOrderSupport ( IDTYPE IdType );


/*******************************************************************************
**
**  Function:  Test5_NoDTC
**
**  Purpose:   Run test 5 with no malfunction, no DTC set
**
*******************************************************************************/
STATUS Test5_NoDTC ( void )
{
	BOOL          bSubTestFailed = FALSE;


//*******************************************************************************
//  Test 5.1 - Do MIL light check
//*******************************************************************************
	gTestSubsection = 1;
	gbTestSubsectionFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(MIL bulb check, Engine Off, No DTC set)" );

	if ( CheckMIL ( ) != PASS || gbTestSubsectionFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.2 - Determine the OBD protocol to use, Engine Off
//*******************************************************************************
	gTestSubsection = 2;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Determine Protocol, Ignition On, Engine Off)" );

	// Determine the OBD protocol to use
	if ( DetermineOBDProtocol ( ) != PASS )
	{
		Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "Protocol determination unsuccessful.\n" );
		Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
		return FAIL;
	}

	if ( VerifyAllINF10Data ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	gbIgnoreUnsupported = TRUE;
	if ( VerifyVehicleState (gbEngineRunning, gbHybrid) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbIgnoreUnsupported = FALSE;

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.3 - Clear codes, Engine Off
//*******************************************************************************
	gTestSubsection = 3;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Clear DTCs (SID $14), Engine Off)" );

	// Clear Codes Functionally Addressed
	if ( ClearCodes ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// Clear Codes Physically Addressed
	gbPhysicalAddressing = TRUE;
	if ( ClearCodes ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbPhysicalAddressing = FALSE;

	gbIgnoreUnsupported = TRUE;
	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbIgnoreUnsupported = FALSE;

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.4 - Verify MIL and DTC status is cleared, Engine Off
//*******************************************************************************
	gTestSubsection = 4;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify MIL status bit, Verify no Permanent DTCs, Engine Off)" );

	// check for response to OBD SID $1 PID $00
	if ( VerifyDeprecatedServices ( 0x01 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	gbIgnoreUnsupported  = TRUE;
	// check MIL Data Functionally Addressed
	if ( VerifyMILData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// check MIL Data Physically Addressed
	gbPhysicalAddressing = TRUE;
	if ( VerifyMILData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbPhysicalAddressing = FALSE;
	gbIgnoreUnsupported  = FALSE;

	// check for response to OBD SID $A
	if ( VerifyDeprecatedServices ( 0x0A ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	// check Permanent DTCs Functionally Addressed
	if ( VerifyPermanentDTCData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// check Permanent DTCs Physically Addressed
	gbPhysicalAddressing = TRUE;
	if ( VerifyPermanentDTCData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbPhysicalAddressing = FALSE;

	gbIgnoreUnsupported = TRUE;
	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbIgnoreUnsupported = FALSE;

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.5 - Verify monitor test support and results, Engine Off
//*******************************************************************************
	gTestSubsection = 5;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify On-Board Monitoring Test Results (SID $19 LEV $1A DTCEDRN $92), Engine Off)" );

	// check for response to OBD SID $6 PID $00
	if ( VerifyDeprecatedServices ( 0x06 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
	Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "Request of Test Results NOT CURRENTLY IMPLEMENTED" );
	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
//	// Get list of DTCs supporting ExtData $92 (Test Result), Functionally Addressed
//	if ( VerifyDTCExtDataList ( 0x92 ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	// Get list of DTCs supporting ExtData $92 (Test Result), Physically Addressed
//	gbPhysicalAddressing = TRUE;
//	if ( VerifyDTCExtDataList ( 0x92 ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	gbPhysicalAddressing = FALSE;
//
//	gbIgnoreUnsupported = TRUE;
//	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	gbIgnoreUnsupported = FALSE;

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.6 - Verify diagnostic support and data, Engine Off
//*******************************************************************************
	gTestSubsection = 6;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Diagnostic Data (SID $22 PIDs), Engine Off)" );

	// check for response to OBD SID $1 PID $00
	if ( VerifyDeprecatedServices ( 0x01 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	// check PID Data Functionally Addressed
	if ( VerifyPIDSupportAndData ( FALSE ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// check PID Data Physically Addressed
	if ( !gbSkipPA )
	{
		gbPhysicalAddressing = TRUE;
		if ( VerifyPIDSupportAndData ( FALSE ) != PASS )
		{
			bSubTestFailed = TRUE;
		}
		gbPhysicalAddressing = FALSE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.7 - Verify control support and data, Engine Off
//*******************************************************************************
	gTestSubsection = 7;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Control of System, Test or Component (SID $31), Engine Off)" );

	// check for response to OBD SID $8 PID $00
	if ( VerifyDeprecatedServices ( 0x08 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
	Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "Request of Test Routine Identifers NOT CURRENTLY IMPLEMENTED" );
	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
//	if ( VerifyControlSupportAndData ( ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//
//	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.8 - Determine Protocol, Engine Running
//*******************************************************************************
	gTestSubsection = 8;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Determine Protocol, Engine Running)" );

	ClearKeyboardBuffer ( );

	Log ( PROMPT, SCREENOUTPUTON, LOGOUTPUTON, ENTER_PROMPT,
	      "Start engine or activate propulsion system for Hybrid/PHEV.\n\n"

	      "Let idle for one (1) minute.\n\n"

	      "Note: Some powertrain control systems have engine controls that can start and\n"
	      "stop the engine without regard to ignition position.\n" );

	gbEngineRunning = TRUE;

	// Determine the OBD protocol to use
	if ( DetermineOBDProtocol ( ) != PASS )
	{
		Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "Protocol determination unsuccessful.\n" );
		Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
		return FAIL;
	}

	// check for response to OBD SID $1 PID $00
	if ( VerifyDeprecatedServices ( 0x01 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.9 - Clear codes, Engine Running
//*******************************************************************************
	gTestSubsection = 9;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Clear DTCs (SID $14), Engine Running)" );

	if ( ClearCodes ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.10 - Verify diagnostic support and data, Engine Running
//*******************************************************************************
	gTestSubsection = 10;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Diagnostic Data (SID $22 PIDs), Engine Running)" );

	// check for response to OBD SID $1 PID $00
	if ( VerifyDeprecatedServices ( 0x01 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyPIDSupportAndData ( FALSE ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.11 - Verify freeze frame support and data, Engine Running
//*******************************************************************************
	gTestSubsection = 11;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Freeze Frame Data (SID $19 LEV $04), Engine Running)" );

	// check for response to OBD SID $2
	if ( VerifyDeprecatedServices ( 0x02 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
	Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "Verify Freeze Frame Data NOT CURRENTLY IMPLEMENTED" );
	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
//	// check FF Data Functionally Addressed
//	if ( VerifyFFSupportAndData ( ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	// check FF Data Physically Addressed
//	gbPhysicalAddressing = TRUE;
//	if ( VerifyFFSupportAndData ( ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	gbPhysicalAddressing = FALSE;
//
//	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
//  Test 5.12 - Verify stored DTC data, Engine Running
//*******************************************************************************
	gTestSubsection = 12;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Stored DTCs (SID $19 LEV $42 StatusMask $08), Engine Running)" );

	// check for response to OBD SID $3
	if ( VerifyDeprecatedServices ( 0x03 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	// check Confirmed DTCs Functionally Addressed
	if ( VerifyConfirmedDTCData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// check Confirmed DTCs Physically Addressed
	gbPhysicalAddressing = TRUE;
	if ( VerifyConfirmedDTCData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbPhysicalAddressing = FALSE;

	if ( VerifyMILData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.13 - Verify monitor test support and results, Engine Running
//*******************************************************************************
	gTestSubsection = 13;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify On-Board Monitoring Test Results (SID $19 LEV $1A DTCEDRN $92), Engine Running)" );

	// check for response to OBD SID $6
	if ( VerifyDeprecatedServices ( 0x06 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
	Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "Request of Test Results NOT CURRENTLY IMPLEMENTED" );
	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
//	// Get list of DTCs supporting ExtData $92 (Test Result), Functionally Addressed
//	if ( VerifyDTCExtDataList ( 0x92 ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	// Get list of DTCs supporting ExtData $92 (Test Result), Physically Addressed
//	gbPhysicalAddressing = TRUE;
//	if ( VerifyDTCExtDataList ( 0x92 ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//	gbPhysicalAddressing = FALSE;
//
//	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.14 - Verify pending DTC data, Engine Running
//*******************************************************************************
	gTestSubsection = 14;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Pending DTCs (SID $19 LEV $42 StatusMask $04), Engine Running)" );

	// check for response to OBD SID $7
	if ( VerifyDeprecatedServices ( 0x07 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	// check Pending DTCs Functionally Addressed
	if ( VerifyPendingDTCData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// check Pending DTCs Physically Addressed
	gbPhysicalAddressing = TRUE;
	if ( VerifyPendingDTCData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	gbPhysicalAddressing = FALSE;

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.15 - Verify control support and data, Engine Running
//*******************************************************************************
	gTestSubsection = 15;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Control of System, Test or Component (SID $31), Engine Running)" );

	// check for response to OBD SID $8
	if ( VerifyDeprecatedServices ( 0x08 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
	Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "Request Control of On-Board System NOT CURRENTLY IMPLEMENTED" );
	Log ( BLANK, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "\n\n" );
//	if ( VerifyControlSupportAndData ( ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}
//
//	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
//	{
//		bSubTestFailed = TRUE;
//	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.16 - Verify vehicle information support and data, Engine Running
//*******************************************************************************
	gTestSubsection = 16;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Vehicle Information (SID $22 INFs), Engine Running)" );

	// check for response to OBD SID $9
	if ( VerifyDeprecatedServices ( 0x09 ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	// check INF Data Functionally Addressed
	if ( VerifyINFSupportAndData ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}
	// check INF Data Physically Addressed
	if ( !gbSkipPA )
	{
		gbPhysicalAddressing = TRUE;
		if ( VerifyINFSupportAndData ( ) != PASS )
		{
			bSubTestFailed = TRUE;
		}
		gbPhysicalAddressing = FALSE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.17 - Verify reverse order support
//*******************************************************************************
	gTestSubsection = 17;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Diagnostic Data (SID $22 PIDs) in reverse order)" );

	if ( VerifyReverseOrderSupport ( PIDREQUEST ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyReverseOrderSupport ( PF5REQUEST ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.18 - Verify tester present support
//*******************************************************************************
	gTestSubsection = 18;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Diagnostic Data (SID $22) idle message timing)" );

	// Sleep for 15 seconds, then make sure the tester present messages kept the link alive
//	Sleep ( 15000 );
	{
		unsigned long ulStartTimeMsecs = GetTickCount ( );
		unsigned long ulTimeMsecs;
		unsigned long ulLastPrintTimeMsecs = 0;
		while ( (ulTimeMsecs = (GetTickCount ( ) - ulStartTimeMsecs)) < 15000)
		{
			if ( (ulTimeMsecs - ulLastPrintTimeMsecs) > 1000 )
			{
				printf ( "INFORMATION: Idle Time Remaining = %d sec  \r", (15000 - ulTimeMsecs)/1000 );
				ulLastPrintTimeMsecs = (GetTickCount ( ) - ulStartTimeMsecs);
			}
		}
		printf ( "\n" );
	}

	// Link active test to verify communication remained active for ALL protocols
	if ( VerifyECUCommunication ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	 {
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	 }
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.19 - Verify diagnostic burst support
//*******************************************************************************
	gTestSubsection = 19;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Diagnostic Data (SID $22) burst message timing)" );

	if ( VerifyPIDBurstSupport ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}


//*******************************************************************************
// Test 5.20 - Verify Reserved Services
//*******************************************************************************
	gTestSubsection = 20;
	gbTestSubsectionFailed = FALSE;
	bSubTestFailed = FALSE;

	Log ( SUBSECTION_BEGIN, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "(Verify Reserved/Unused Services, Engine Running)" );

	if ( VerifyReservedServices ( ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( VerifyVehicleState ( gbEngineRunning, gbHybrid ) != PASS )
	{
		bSubTestFailed = TRUE;
	}

	if ( gbTestSubsectionFailed == TRUE || bSubTestFailed == TRUE )
	{
		if ( Log ( SUBSECTION_FAILED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT, "" ) == 'N' )
		{
			return FAIL;
		}
	}
	else
	{
		Log ( SUBSECTION_PASSED_RESULT, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT, "" );
	}

	return PASS;
}
