/*******************************************************************************
********************************************************************************
**
**  Copyright(c) 2022, Alliance for Automotive Innovation
**  Used only under license from the Alliance for Automotive Innovation. All Rights Reserved.
**
**  Project:  J1699-5
**  FileName: VerifyTIDSupportAndData.c
**  Author:   EnGenius
**  Date:     2/25/2022
**  Email:    <support@autosinnovate.org>
**
**  Purpose:  SAE J1699-5 Vehicle OBD II Compliance Test Cases Source Code.
**            This source code is intended to run the tests described in
**            the SAE J1699-5 document in an automated manner, when compiled
**            and used with an SAE J2534-compatible pass-thru device.
**
**            File j1699.c contains information on building and running this test.
**
**  Description:
**
**  Modifications:  03/13/2023  Initial Version
**
********************************************************************************
*******************************************************************************/

#include <stdio.h>    // C Library input and output declarations
#include <stdlib.h>   // C Library general function declarations
#include <time.h>     // C Library time and date declarations
#include <windows.h>  // Windows API declarations
#include "j2534.h"    // j1699 project j2534 declarations
#include "j1699.h"    // j1699 project general declarations


STATUS RequestSID8SupportData ( void );
void   FormatSID8SupReq ( unsigned int IdIdx, REQ_MSG *stReqMsg );
STATUS VerifySID8PIDSupportData ( void );


/*******************************************************************************
**
**  Function:  VerifyTIDSupportAndData -
**
**  Purpose:   Verify SID 8 control support and data
**
*******************************************************************************/
STATUS VerifyTIDSupportAndData ( void )
{
	STATUS        eRetCode;

	unsigned long InitialFailureCount;


	InitialFailureCount = GetFailureCount ( );

	// Request SID 8 support data
	eRetCode = RequestIDSupportData ( TIDREQUEST, TRUE );
	if ( eRetCode == FAIL )
	{
		return FAIL;
	}

	if ( eRetCode == ABORT )
	{
		return PASS;
	}

	if ( RequestGroupPIDSupportData ( TIDREQUEST ) == FAIL )
	{
		return FAIL;
	}

	// Link active test to verify communication remained active for ALL protocols
	if ( VerifyECUCommunication ( ) != PASS )
	{
		return FAIL;
	}

	if ( InitialFailureCount != GetFailureCount ( ) )
	{
		// There could have been early/late responses that weren't treated as FAIL
		return FAIL;
	}
	else
	{
		return PASS;
	}

}


/*******************************************************************************
**
**  FUNCTION:     FormatSID8SupReq
**
**  DESCRIPTION:  Routine
**
*******************************************************************************/
void FormatSID8SupReq ( unsigned int IdIdx, REQ_MSG *stReqMsg )
{

	stReqMsg->SID = 8;                       // SID $08

	stReqMsg->u.DID[0] = IdIdx;  // Test ID (request Test ID values)

	// Documented in ISO15031-5, "ISO_DIS_15031-5.4_(E)_RevOn_2004-01-20.doc"
	// Annex F
	//"For ISO 15765-4 protocol DATA_A - DATA_E shall not be included in the
	// request and response message."

	stReqMsg->NumIds = 1;
}


/*
******************************************************************************
**
**  Function:  RequestSID8SupportData
**
**  Purpose:   Purpose of this routine is to verify that SID 8 PID 00
**             returns a support record. Continue requesting support
**             PIDs thru the highest supported group.
**
********************************************************************************
*/
STATUS RequestSID8SupportData ( void )
{
	unsigned int  IdIdx;
	BYTE          EcuIdx;
	unsigned long ulTIDSupport;  // Evaluate $E0 TID support indication.
	REQ_MSG       stReqMsg;

	// Request SID 8 support data
	for ( IdIdx = 0x00;
	      IdIdx < 0x100;
	      IdIdx += 0x20 )
	{
		FormatSID8SupReq ( IdIdx, &stReqMsg );

		if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL|REQ_MSG_ALLOW_NO_RESPONSE ) == FAIL )
		{
			Log ( WARNING, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "No response to SID $8 TID-Supported TID %02X request\n",
			      IdIdx );
			return ABORT;
		}

		// Check if we need to request the next group
		for ( EcuIdx = 0;
		      EcuIdx < gNumOfECUs;
		      EcuIdx++ )
		{
			if ( gstResponse[EcuIdx].TIDSupport[IdIdx >> 5].SupBits[3] & 0x01 )
			{
				break;
			}
		}
		if ( EcuIdx >= gNumOfECUs )
		{
			break;
		}
	}

	// Flag error if ECU indicates no support
	if ( VerifySID8PIDSupportData ( ) == FAIL )
	{
		return FAIL;
	}

	// call for request of next unsupported TID support TID

	// Enhance logic to verify support information if request is at upper limit of $E0
	if ( IdIdx == 0xE0 )
	{
		// Init variable to no-support
		ulTIDSupport = 0;

		// For each ECU
		for ( EcuIdx = 0;
		      EcuIdx < gNumOfECUs;
		      EcuIdx++ )
		{
			// If MID is supported, keep looking
			if ( ( gstResponse[EcuIdx].TIDSupport[IdIdx >> 5].SupBits[0]        ||
			       gstResponse[EcuIdx].TIDSupport[IdIdx >> 5].SupBits[1]        ||
			       gstResponse[EcuIdx].TIDSupport[IdIdx >> 5].SupBits[2]        ||
			     ( gstResponse[EcuIdx].TIDSupport[IdIdx >> 5].SupBits[3] & 0xFE ) ) != 0x00)
			{
				// Flag as support indicated!
				ulTIDSupport = 1;
			}
		}
		// Flag as error if no support indicated in $E0
		if ( ulTIDSupport == 0x00 )
		{
			if ( Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT,
			           "No SID $8 TID $E0 support indicated!\n" ) == 'N' )
			{
				return FAIL;
			}
		}
	}
	else
	{
		FormatSID8SupReq ( (IdIdx += 0x20), &stReqMsg );

		if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL|REQ_MSG_IGNORE_NO_RESPONSE ) != FAIL )
		{
			for ( EcuIdx = 0;
			      EcuIdx < gUserNumOfECUs;
			      EcuIdx++ )
			{
				if ( gstResponse[EcuIdx].TIDSize != 0 )
				{
					Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
					      "ECU %X  Unexpected response to unsupported SID $8 TID $%02X!\n",
					      GetEcuId ( EcuIdx ),
					      IdIdx );
				}
			}
			return FAIL;
		}
	}

	// Verify ECU did not drop out.
	if ( VerifyECUCommunication ( ) != PASS )
	{
		return FAIL;
	}

	return PASS;
}


/*
*****************************************************************************
**  Function:  VerifySID8PIDSupportData
**
**  Purpose:   Verify each controller supports at a minimum one PID.
**             Any ECU that responds that does not support at least
**             one PID is flagged as an error.
**
*****************************************************************************
*/
STATUS VerifySID8PIDSupportData ( void )
{
	STATUS          eReturn = PASS;
	STATUS          eEcuResult;
	BYTE            EcuIdx;
	unsigned long   Index;

	// For each ECU
	for ( EcuIdx = 0;
	      EcuIdx < gNumOfECUs;
	      EcuIdx++ )
	{
		eEcuResult = FAIL;
		for ( Index = 0;
		      Index < gstResponse[EcuIdx].TIDSupportSize;
		      Index++ )
		{
			// If MID is supported, keep looking
			if ( gstResponse[EcuIdx].TIDSupport[Index].SupBits[0] ||
			     gstResponse[EcuIdx].TIDSupport[Index].SupBits[1] ||
			     gstResponse[EcuIdx].TIDSupport[Index].SupBits[2] ||
			     (gstResponse[EcuIdx].TIDSupport[Index].SupBits[3] & 0xFE) != 0x00 )
			{
				eEcuResult = PASS;
				break;
			}
		}

		if ( eEcuResult == FAIL &&
		     gstResponse[EcuIdx].TIDSupportSize > 0 )
		{
			if ( Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, YES_NO_ALL_PROMPT,
			           "ECU %X  SID $8 TID-Supported TIDs indicate no TIDS supported\n",
			           GetEcuId ( EcuIdx ) ) == 'N' )
			{
				eReturn = FAIL;
			}
		}
	}

	return eReturn;
}


/*
*******************************************************************************
**  VerifyGroupControlSupport -
**  Function to verify SID8 group support
*******************************************************************************
*/
STATUS VerifyGroupControlSupport ( void )
{
//	BYTE    EcuIdx;
//	REQ_MSG stReqMsg;
//
//	BOOL bTestFailed = FALSE;
//
//	int bReqNextGroup;      // Binary logic for request for next group!
//
//
//	// Save the original response data for comparison
//	memcpy ( &gstCompareResponse[0],
//	         &gstResponse[0],
//	         sizeof ( ECU_DATA ) * gNumOfECUsForAlloc );
//
//	// Request SID 31 support data as a group
//	stReqMsg.SID     = 0x31;
//	stReqMsg.NumIds  = 1;
//	stReqMsg.u.ID[0] = 0x00;
//	stReqMsg.u.ID[1] = 0x20;
//	stReqMsg.u.ID[2] = 0x40;
//	stReqMsg.u.ID[3] = 0x60;
//	stReqMsg.u.ID[4] = 0x80;
//	stReqMsg.u.ID[5] = 0xA0;
//	if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL|REQ_MSG_ALLOW_NO_RESPONSE ) == FAIL )
//	{
//		Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
////		      "SID $31 group ($00-$A0) support request failed\n" );
//		      "SID $31 request failed\n" );
//		bTestFailed = TRUE;
//	}
//	else
//	{
//		// Set next group select to false
//		bReqNextGroup = 0;
//
//		// Determine if next group support request is required
//		for ( EcuIdx = 0;
//		      EcuIdx < gNumOfECUs;
//		      EcuIdx++ )
//		{
//			// Index 0x05 corresponds to 0xA0
//			if ( gstResponse[EcuIdx].TIDSupport[5].SupBits[3] & 0x01 )
//			{
//				bReqNextGroup = 1;
//			}
//		}
//
//		// Request next group if supported by previous request!
//		// Test support for $C0 & $E0
//		if ( bReqNextGroup == 1 )
//		{
//			// Request SID 8 support data as a group
//			stReqMsg.SID     = 31;
//			stReqMsg.NumIds  = 2;
//			stReqMsg.u.ID[0] = 0xC0;
//			stReqMsg.u.ID[1] = 0xE0;
//			if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL ) == FAIL )
//			{
//				Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
//				      "SID $8 group ($C0-$E0) support request failed\n" );
//				bTestFailed = TRUE;
//			}
//		}
//
//		// Verify that all SID 8 TID group/individual support data compares
//		for ( EcuIdx = 0;
//		      EcuIdx < gNumOfECUs;
//		      EcuIdx++ )
//		{
//			if ( CompareIDSupport ( EcuIdx,
//			                        0x08,
//			                        TRUE ) == FAIL )
//			{
//				bTestFailed = TRUE;
//			}
//		}
//	}
//
//	if ( bTestFailed == TRUE )
//	{
//		// Something was wrong, restore the original response data
//		memcpy ( &gstResponse[0],
//		         &gstCompareResponse[0],
//		         sizeof ( ECU_DATA ) * gNumOfECUsForAlloc );
//		return FAIL;
//	}

	return PASS;
}
