/*******************************************************************************
********************************************************************************
**
**  Copyright(c) 2022, Alliance for Automotive Innovation
**  Used only under license from the Alliance for Automotive Innovation. All Rights Reserved.
**
**  Project:  J1699-5
**  FileName: Test10_InUseCounters.c
**  Author:   EnGenius
**  Date:     2/25/2022
**  Email:    <support@autosinnovate.org>
**
**  Purpose:  SAE J1699-5 Vehicle OBD II Compliance Test Cases Source Code.
**            This source code is intended to run the tests described in
**            the SAE J1699-5 document in an automated manner, when compiled
**            and used with an SAE J2534-compatible pass-thru device.
**
**            File j1699.c contains information on building and running this test.
**
**  Description:
**
**  Modifications:  03/04/2024  Initial Version
**
********************************************************************************
*******************************************************************************/

#include <stdio.h>    // C Library input and output declarations
#include <stdlib.h>   // C Library general function declarations
#include <string.h>   // C Library character array declarations
#include <time.h>     // C Library time and date declarations
#include <conio.h>    // MS-DOS console input and output declarations
#include <windows.h>  // Windows API declarations
#include "j2534.h"    // j1699 project j2534 declarations
#include "j1699.h"    // j1699 project general declarations


STATUS VerifyDTCExtDataList ( BYTE  ExtDataId );
void   LogDTCIUMPRList      ( void );
void   LogDTCSMADList       ( void );

STATUS VerifyDTCExtData     ( BYTE  ExtDataID );
STATUS GetDTCExtData        ( BYTE  ExtDataID,
                              BYTE  Index );
STATUS SaveDTCIUMPRData     ( BYTE  EcuIdx,
                              BYTE *pIndex );
STATUS SaveDTCSMADData      ( BYTE  EcuIdx,
                              BYTE *pIndex );
void   LogDTCIUMPRData      ( BYTE  EcuIdx,
                              BYTE  Index );
void   LogDTCSMADData       ( BYTE  EcuIdx,
                              BYTE  Index );

STATUS VerifyDTCIMReadinessList ( void );
void SaveDTCIMReadinessList     ( BYTE  EcuIdx );
void LogDTCIMReadinessList      ( BYTE  EcuIdx,
                                  BYTE  Index );


extern const char szEcuId[];

const char szIumprListSize[]  =   "DTCIUMPRLIST SIZE";
const char szIumprSize[]      =   "IUMPR SIZE";
const char szIumprName[]      =   "DTC";
const char szIumprData1[]     =   "NUMERATOR";
const char szIumprData2[]     =   "DENOMINATOR";

const char szSmadListSize[]   =   "DTCSMADLIST SIZE";
const char szSmadSize[]       =   "SMAD SIZE";
const char szSmadName[]       =   "DTC";
const char szSmadData1[]      =   "MINI NUMERATOR";
const char szSmadData2[]      =   "MINI DENOMINATOR";
const char szSmadData3[]      =   "ACTIVITY RATIO";

const char szUnassigned[]  =   "UNASSIGNED";

const char *szIMReadinessGroup [] =
{
	"No Readiness Group",
	"CAT  Catalyst Monitor",
	"HCAT Heated Catalyst Monitor",
	"MIS  Misfire Monitor",
	"EVAP Evaporative System Monitoring",
	"AIR  Secondary Air System Monitoring",
	"FUEL Fuel System Monitoring",
	"EGS  Exhaust Gas Sensor Monitoring",
	"EGR  Exhaust Gas Recirculation System Monitoring",
	"CV   Positive/Crankcase Ventilation System Monitoring",
	"COOL Engine Cooling System Monitoring",
	"CSER Cold Start Emission Reduction Strategy Monitoring",
	"VVT  Variable Valve Timing, Lift, and/or Control System Monitoring",
	"DOR  Direct Ozone Reduction System Monitoring",
	"CCM  Comprehensive Component Monitoring",
	"OTH  Other Emission Control or Source System Monitoring",
	"HCCAT Non-Methane Hydrocarbon Converting Catalyst Monitoring",
	"NCAT Oxides of Nitrogen Converting Catalyst Monitoring",
	"BP   Boost Pressure Control System Monitoring",
	"NAC  NOx Adsorber Monitoring",
	"PF   Particulate Matter Filter Monitoring"
};
#define MAX_IM_READINESS_GROUP 0x14


/*******************************************************************************
**
**  Function:  VerifyDTCExtDataList
**
**  Purpose:   
**
*******************************************************************************/
STATUS VerifyDTCExtDataList ( BYTE ExtDataID )
{
	REQ_MSG        stReqMsg;
	STATUS         eRetCode = PASS;


	Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
	      "Request DTC Extended Data list (SID $19 LEV $1A DTCEDRN $%02X) %s Addressed\n",
	      ExtDataID,
	      gbPhysicalAddressing ? "Physically" : "Functionally" );

	stReqMsg.SID     = 0x19;
	stReqMsg.NumIds  = 2;
	stReqMsg.u.ID[0] = 0x1A;       // ReportDTCExtDataIdentification
	stReqMsg.u.ID[1] = ExtDataID;  // DTCExtendedDTCDataRecordNumber
	if ( (gbPhysicalAddressing == FALSE && (eRetCode = RequestSID ( &stReqMsg, REQ_MSG_NORMAL )) == FAIL ) ||
	     (gbPhysicalAddressing == TRUE  && (eRetCode = RequestSID_PhysicallyAddressed_All ( &stReqMsg, REQ_MSG_NORMAL)) == FAIL ))
	{
		Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "SID $19 $1A $%02X request failed\n",
		      ExtDataID );
		return FAIL;
	}

	// Display the responses
	if ( ExtDataID == 0x91 )
	{
		LogDTCIUMPRList ( );
	}
	else
	{
		LogDTCSMADList ( );
	}

	return eRetCode;
}


/*******************************************************************************
**
**  Function:  LogDTCIUMPRList
**
**  Purpose:
**
*******************************************************************************/
void LogDTCIUMPRList ( void )
{
	BYTE     EcuIdx;

	BYTE     EcuDTCIndex;
	BYTE     ListDTCIndex;


	for ( EcuIdx = 0;
	      EcuIdx < gNumOfECUs;
	      EcuIdx++ )
	{
		if ( gstResponse[EcuIdx].bDTCIUMPRSupported != TRUE )
		{
			Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "ECU %-8X  supports no DTCs with Extdata $91.\n",
			      GetEcuId ( EcuIdx ) );
		}
		else
		{
			Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "ECU %-8X  supports the following DTCs with Extdata $91:\n",
			      GetEcuId ( EcuIdx ) );

			// log IUMPR names and values
			for ( EcuDTCIndex = 0;
			      EcuDTCIndex < gstResponse[EcuIdx].DTCIUMPRCount;
			      EcuDTCIndex++ )
			{

				Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
				      "              %c%02X%02X (Failure Type = $%02X  Status = $%02X)\n",
				       DTCTypeCharacter[(gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
				       gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
				       gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
				       gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.LowByte,                                   // Failure Type Byte
				       gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Status );                                      // Status of DTC

				// check if the DTC is already in the list
				for ( ListDTCIndex = 0;
				      ListDTCIndex < DTCIUMPRListCount;
				      ListDTCIndex++ )
				{
					if ( DTCIUMPRList[ListDTCIndex].HighByte == gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.HighByte &&
					     DTCIUMPRList[ListDTCIndex].MidByte  == gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.MidByte &&
					     DTCIUMPRList[ListDTCIndex].LowByte  == gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.LowByte )
					{
						break;
					}
				}

				// if the DTC wasn't in the list, add it
				if ( ListDTCIndex == DTCIUMPRListCount )
				{
					if ( DTCIUMPRListCount < MAX_DTCIUMPR_COUNT )
					{
						DTCIUMPRList[ListDTCIndex].HighByte = gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.HighByte;
						DTCIUMPRList[ListDTCIndex].MidByte  = gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.MidByte;
						DTCIUMPRList[ListDTCIndex].LowByte  = gstResponse[EcuIdx].DTCIUMPRList[EcuDTCIndex].Record.Dtc.LowByte;
						DTCIUMPRListCount++;
					}
					else
					{
						Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
						      "More than the allocated number of DTCs with Extdata $91:\n" );
					}

				}
			}
		}
	}
}


/*******************************************************************************
**
**  Function:  LogDTCSMADList
**
**  Purpose:
**
*******************************************************************************/
void LogDTCSMADList ( void )
{
	BYTE     EcuIdx;
	BYTE     EcuDTCIndex;
	BYTE     DTCIndex;

	for ( EcuIdx = 0;
	      EcuIdx < gNumOfECUs;
	      EcuIdx++ )
	{
		if ( gstResponse[EcuIdx].bDTCSMADSupported  != TRUE )
		{
			Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "ECU %-8X  supports no DTCs with Extdata $93.\n",
			      GetEcuId ( EcuIdx ) );
		}
		else
		{
			Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "ECU %-8X  supports the following DTCs with Extdata $93:\n",
			      GetEcuId ( EcuIdx ) );

			// log IUMPR names and values
			for ( EcuDTCIndex = 0;
			      EcuDTCIndex < gstResponse[EcuIdx].DTCSMADCount;
			      EcuDTCIndex++ )
			{

				Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
				      "              %c%02X%02X (Failure Type = $%02X  Status = $%02X)\n",
				       DTCTypeCharacter[(gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
				       gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
				       gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
				       gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.LowByte,                                   // Failure Type Byte
				       gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Status );                                      // Status of DTC

				// check if the DTC is already in the list
				for ( DTCIndex = 0;
				      DTCIndex < DTCSMADListCount;
				      DTCIndex++ )
				{
					if ( DTCSMADList[DTCIndex].HighByte == gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.HighByte &&
					     DTCSMADList[DTCIndex].MidByte  == gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.MidByte &&
					     DTCSMADList[DTCIndex].LowByte  == gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.LowByte )
					{
						break;
					}
				}

				// if the DTC wasn't in the list, add it
				if ( DTCIndex == DTCSMADListCount )
				{
					DTCSMADList[DTCIndex].HighByte = gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.HighByte;
					DTCSMADList[DTCIndex].MidByte  = gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.MidByte;
					DTCSMADList[DTCIndex].LowByte  = gstResponse[EcuIdx].DTCSMADList[EcuDTCIndex].Record.Dtc.LowByte;
					DTCSMADListCount++;
				}
			}
		}
	}
}


/*******************************************************************************
**
**  Function:  VerifyDTCExtData
**
**  Purpose:   
**
*******************************************************************************/
STATUS VerifyDTCExtData ( BYTE ExtDataId )
{
	BYTE       EcuIdx;
	STATUS     eRetCode = PASS;

	STATUS     eCode = PASS;

	char      *pString = (char*)&szUnassigned;

	BYTE       Index;
	BYTE       IndexMax = 0;

	BYTE       DTCIndex;


	if ( ExtDataId == 0x91 )
	{
		IndexMax = DTCIUMPRListCount;
	}
	else if ( ExtDataId == 0x93 )
	{
		IndexMax = DTCSMADListCount;
	}

	// Request all DTC Extended Data
	for ( Index = 0;
	      Index < IndexMax;
	      Index++ )
	{
		// Request the DTC Extended Data
		if ( (eCode = GetDTCExtData ( ExtDataId, Index )) == PASS )
		{
			for ( EcuIdx = 0;
			      EcuIdx < gNumOfECUs;
			      EcuIdx++ )
			{
				// if a response was received from this ECU
				if ( gstResponse[EcuIdx].bResponseReceived  == TRUE )
				{
					// Save and log the response
					if ( ExtDataId == 0x91 &&
					     gstResponse[EcuIdx].bDTCIUMPRSupported == TRUE )
					{
						if ( (eCode = SaveDTCIUMPRData ( EcuIdx, &DTCIndex )) == PASS )
						{
							// if not TEST 10.10, log each response individually
							if ( !(geTestPhase == eTestInUseCounters && gTestSubsection == 10) )
							{
								LogDTCIUMPRData ( EcuIdx, DTCIndex );
							}
						}
						else if ( eRetCode == PASS )
						{
							eRetCode |= eCode;
						}
						continue;
					}
					else if ( ExtDataId == 0x93 &&
					          gstResponse[EcuIdx].bDTCSMADSupported  == TRUE )
					{
						if ( (eRetCode = SaveDTCSMADData ( EcuIdx, &DTCIndex )) == PASS )
						{
							// if not TEST 10.10, log each response individually
							if ( !(geTestPhase == eTestInUseCounters && gTestSubsection == 10) )
							{
								LogDTCSMADData ( EcuIdx, DTCIndex );
							}
						}
						else if ( eRetCode == PASS )
						{
							eRetCode |= eCode;
						}
						continue;
					}
				}
			}  // end for ( EcuIdx )
		}  // end if ( GetDTCExtData () == PASS )
		else if ( eRetCode == PASS )
		{
			eRetCode |= eCode;
		}
	}  // end for ( Index )

	// if TEST 10.10,for each ECU, log all the IUMPR and SMAD data as a group
	if ( geTestPhase == eTestInUseCounters && gTestSubsection == 10 )
	{
		for ( EcuIdx = 0;
		      EcuIdx < gNumOfECUs;
		      EcuIdx++ )
		{
			// log ECU ID
			Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "%-10s = 0x%X\n",
			      szEcuId,
			      GetEcuId ( EcuIdx ) );

			if ( ExtDataId == 0x91 )
			{
				IndexMax = gstResponse[EcuIdx].DTCIUMPRCount;
				pString = (char *)&szIumprSize[0];

			}
			else if ( ExtDataId == 0x93 )
			{
				IndexMax = gstResponse[EcuIdx].DTCSMADCount;
				pString = (char *)&szSmadSize[0];
			}

			// log number of data items
			Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "%-10s = %d\n",
			      pString,
			      IndexMax );

			for ( Index = 0;
			      Index < IndexMax;
			      Index++ )
			{
				if ( ExtDataId == 0x91 && gstResponse[EcuIdx].bDTCIUMPRSupported == TRUE )
				{
					LogDTCIUMPRData ( EcuIdx, Index );
				}
				else if ( ExtDataId == 0x93 && gstResponse[EcuIdx].bDTCSMADSupported  == TRUE )
				{
					LogDTCSMADData ( EcuIdx, Index );
				}
			}  // end for ( Index )
		}  // end for ( EcuIdx )
	}  // end if ( TEST 10.14 )

	return eRetCode;
}


/*******************************************************************************
**
**  Function:  GetDTCExtData
**
**  Purpose:   Copy from gstResponse[EcuIdx].DTCIUMPRCurrent into the matching DTC
**             in gstResponse[EcuIdx].DTCIUMPRList.
**
*******************************************************************************/
STATUS GetDTCExtData ( BYTE ExtDataId,
                       BYTE Index )
{
	REQ_MSG    stReqMsg;
	STATUS     eRetCode = PASS;


	// Request the DTC data
	stReqMsg.SID     = 0x19;
	stReqMsg.NumIds  = 5;
	stReqMsg.u.ID[0] = 0x06;                              // ReportDTCExtDataIdentification
	if ( ExtDataId == 0x91 )
	{
		stReqMsg.u.ID[1] = DTCIUMPRList[Index].HighByte;  // High byte of DTC
		stReqMsg.u.ID[2] = DTCIUMPRList[Index].MidByte;   // Mid  byte of DTC
		stReqMsg.u.ID[3] = DTCIUMPRList[Index].LowByte;   // Low  byte of DTC
	}
	else
	{
		stReqMsg.u.ID[1] = DTCSMADList[Index].HighByte;   // High byte of DTC
		stReqMsg.u.ID[2] = DTCSMADList[Index].MidByte;    // Mid  byte of DTC
		stReqMsg.u.ID[3] = DTCSMADList[Index].LowByte;    // Low  byte of DTC
	}
	stReqMsg.u.ID[4] = ExtDataId;                         // DTCExtendedDTCDataRecordNumber
	if ( eRetCode = RequestSID ( &stReqMsg, REQ_MSG_NO_PERIODIC_DISABLE ) == FAIL )
	{
		Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "SID $19 $06 DTC %c%02X%02X (Failure Type = $%02X) ExtData $%02X request failed\n",
		       DTCTypeCharacter[(stReqMsg.u.ID[1] & 0xC0) >> 6],  // 1st character (P, C, B, U)
		       stReqMsg.u.ID[1] & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		       stReqMsg.u.ID[2],                                  // 4th and 5th characters (0-F)
		       stReqMsg.u.ID[3],                                  // Failure Type Byte
		       ExtDataId );                                       // Extended Data ID
		return FAIL;
	}

	return eRetCode;
}


/*******************************************************************************
**
**  Function:  SaveDTCIUMPRData
**
**  Purpose:   Copy from gstResponse[EcuIdx].DTCIUMPRCurrent into the matching DTC
**             in gstResponse[EcuIdx].DTCIUMPRList.
**
*******************************************************************************/
STATUS SaveDTCIUMPRData ( BYTE  EcuIdx,
                          BYTE *pIndex )
{
	STATUS        eRetCode = PASS;  // function result

	DTCIUMPR     *pIUMPR = NULL;


	// find current DTC in DTC IUMPR list and copy the data
	for ( *pIndex = 0;
	      *pIndex < gstResponse[EcuIdx].DTCIUMPRCount;
	      (*pIndex)++ )
	{
		// if the received DTC is in this ECUs list
		if ( gstResponse[EcuIdx].DTCIUMPRList[*pIndex].Record.Dtc.HighByte == gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.HighByte &&
		     gstResponse[EcuIdx].DTCIUMPRList[*pIndex].Record.Dtc.MidByte  == gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.MidByte &&
		     gstResponse[EcuIdx].DTCIUMPRList[*pIndex].Record.Dtc.LowByte  == gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.LowByte )
		{
			// save the data to the ECU's IUMPR list
			gstResponse[EcuIdx].DTCIUMPRList[*pIndex].CompCounts = gstResponse[EcuIdx].DTCIUMPRCurrent.CompCounts;
			gstResponse[EcuIdx].DTCIUMPRList[*pIndex].CondCounts = gstResponse[EcuIdx].DTCIUMPRCurrent.CondCounts;

			// if this is a test phase that needs it's data saved for later testing,
			// also save the data to that phase's copy of the IUMPR list
			if ( (geTestPhase == eTestInUseCounters       && gTestSubsection == 10) ||
			     (geTestPhase == eTestPerformanceCounters && gTestSubsection == 5)  ||
			     (geTestPhase == eTestPerformanceCounters && gTestSubsection == 11) )
			{
				if ( geTestPhase == eTestInUseCounters && gTestSubsection == 10 )
				{
					pIUMPR = &gstResponse[EcuIdx].DTCIUMPR_Test10_10[0];
				}
				else if ( geTestPhase == eTestPerformanceCounters && gTestSubsection == 5 )
				{
					pIUMPR = &gstResponse[EcuIdx].DTCIUMPR_Test11_5[0];
				}
				else if ( geTestPhase == eTestPerformanceCounters && gTestSubsection == 11 )
				{
					pIUMPR = &gstResponse[EcuIdx].DTCIUMPR_Test11_11[0];
				}

				pIUMPR[*pIndex].Record.Dtc.HighByte = gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.HighByte;
				pIUMPR[*pIndex].Record.Dtc.MidByte  = gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.MidByte;
				pIUMPR[*pIndex].Record.Dtc.LowByte  = gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.LowByte;

				pIUMPR[*pIndex].CompCounts = gstResponse[EcuIdx].DTCIUMPRCurrent.CompCounts;
				pIUMPR[*pIndex].CondCounts = gstResponse[EcuIdx].DTCIUMPRCurrent.CondCounts;
			}
			break;
		}  // end if ( matching DTC found )
	}  // end for ( Index )

	// if the current DTC was not in the DTC IUMPR list, warn
	if ( *pIndex >= gstResponse[EcuIdx].DTCIUMPRCount )
	{
		Log ( WARNING, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "ECU %-8X  received unexpected response to SID $19 $06 DTC %c%02X%02X %02X ExtData $91 request (Response not stored)\n",
		       GetEcuId ( EcuIdx ),
		       DTCTypeCharacter[(gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		       gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		       gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		       gstResponse[EcuIdx].DTCIUMPRCurrent.Record.Dtc.LowByte );                                 // Failure Type Byte

		eRetCode = ERRORS;
	}

	return eRetCode;
}


/*******************************************************************************
**
**  Function:  SaveDTCSMADData
**
**  Purpose:   Copy from gstResponse[EcuIdx].DTCSMADCurrent into the matching DTC
**             in gstResponse[EcuIdx].DTCSMADList.
**
*******************************************************************************/
STATUS SaveDTCSMADData ( BYTE  EcuIdx,
                         BYTE *pIndex )
{
	STATUS        eRetCode = PASS;  // function result

	DTCSMAD      *pSMAD = NULL;


	// find current DTC in DTC SMAD list and copy the data
	for ( *pIndex = 0;
	      *pIndex < gstResponse[EcuIdx].DTCSMADCount;
	      (*pIndex)++ )
	{
		if ( gstResponse[EcuIdx].DTCSMADList[*pIndex].Record.Dtc.HighByte == gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.HighByte &&
		     gstResponse[EcuIdx].DTCSMADList[*pIndex].Record.Dtc.MidByte  == gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.MidByte &&
		     gstResponse[EcuIdx].DTCSMADList[*pIndex].Record.Dtc.LowByte  == gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.LowByte )
		{
			// save the data to the main SMAD list
			gstResponse[EcuIdx].DTCSMADList[*pIndex].NumCounts   = gstResponse[EcuIdx].DTCSMADCurrent.NumCounts;
			gstResponse[EcuIdx].DTCSMADList[*pIndex].DenomCounts = gstResponse[EcuIdx].DTCSMADCurrent.DenomCounts;
			gstResponse[EcuIdx].DTCSMADList[*pIndex].ActRatio    = gstResponse[EcuIdx].DTCSMADCurrent.ActRatio;

			// if this is a test phase that needs it's data saved for later testing,
			// also save the data to that phase's copy of the SMAD list
			if ( (geTestPhase == eTestInUseCounters       && gTestSubsection == 10) ||
			     (geTestPhase == eTestPerformanceCounters && gTestSubsection == 5)  ||
			     (geTestPhase == eTestPerformanceCounters && gTestSubsection == 11) )
			{
				if ( geTestPhase == eTestInUseCounters && gTestSubsection == 10 )
				{
					pSMAD = &gstResponse[EcuIdx].DTCSMAD_Test10_10[0];
				}
				else if ( geTestPhase == eTestPerformanceCounters && gTestSubsection == 5 )
				{
					pSMAD = &gstResponse[EcuIdx].DTCSMAD_Test11_5[0];
				}
				else if ( geTestPhase == eTestPerformanceCounters && gTestSubsection == 11 )
				{
					pSMAD = &gstResponse[EcuIdx].DTCSMAD_Test11_11[0];
				}

				pSMAD[*pIndex].Record.Dtc.HighByte = gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.HighByte;
				pSMAD[*pIndex].Record.Dtc.MidByte  = gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.MidByte;
				pSMAD[*pIndex].Record.Dtc.LowByte  = gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.LowByte;

				pSMAD[*pIndex].NumCounts   = gstResponse[EcuIdx].DTCSMADCurrent.NumCounts;
				pSMAD[*pIndex].DenomCounts = gstResponse[EcuIdx].DTCSMADCurrent.DenomCounts;
				pSMAD[*pIndex].ActRatio    = gstResponse[EcuIdx].DTCSMADCurrent.ActRatio;
			}
			break;
		}
	}

	// if the current DTC was not in the DTC SMAD list, warn
	if ( *pIndex >= gstResponse[EcuIdx].DTCSMADCount )
	{
		Log ( WARNING, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "ECU %-8X  received unexpected response to SID $19 $06 DTC %c%02X%02X %02X ExtData $93 request (Response not stored)\n",
		       GetEcuId ( EcuIdx ),
		       DTCTypeCharacter[(gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		       gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		       gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		       gstResponse[EcuIdx].DTCSMADCurrent.Record.Dtc.LowByte );                                 // Failure Type Byte

		eRetCode = ERRORS;
	}

	return eRetCode;
}


/*******************************************************************************
**
**  Function:  LogDTCIUMPRData
**
**  Purpose:
**
*******************************************************************************/
void LogDTCIUMPRData ( BYTE  EcuIdx,
                       BYTE  Index )
{
	char  String[16];


	if ( gstResponse[EcuIdx].bDTCIUMPRSupported == TRUE &&
	     gstResponse[EcuIdx].DTCIUMPRCount != 0 )
	{
		// if not TEST 10.10 or 14 or 11.1, log ECU ID
		if ( !((geTestPhase == eTestInUseCounters       && (gTestSubsection == 10 || gTestSubsection == 14 )) ||
		       (geTestPhase == eTestPerformanceCounters &&  gTestSubsection == 1)) )
		{
			// include ECU ID in log entry
			sprintf_s ( String, sizeof ( String ),
			           "ECU %-8X  ",
			           GetEcuId ( EcuIdx ) );
		}
		else
		{
			// if Test 10.10,14 or 11.1, do not include ECU ID in log entry
			sprintf_s ( String, sizeof ( String ),
			            "" );
		}

		// log IUMPR names and values
		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "%sDTC %c%02X%02X %02X  %-11s = %3d\n",
		      String,
		      DTCTypeCharacter[(gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		      gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		      gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		      gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.LowByte,                                   // Failure Type Byte
		      szIumprData1,
		      gstResponse[EcuIdx].DTCIUMPRList[Index].CompCounts );

		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "%sDTC %c%02X%02X %02X  %-11s = %3d\n",
		      String,
		      DTCTypeCharacter[(gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		      gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		      gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		      gstResponse[EcuIdx].DTCIUMPRList[Index].Record.Dtc.LowByte,                                   // Failure Type Byte
		      szIumprData2,
		      gstResponse[EcuIdx].DTCIUMPRList[Index].CondCounts );
	}
}


/*******************************************************************************
**
**  Function:  LogDTCSMADData
**
**  Purpose:
**
*******************************************************************************/
void LogDTCSMADData ( BYTE  EcuIdx,
                      BYTE  Index )
{
	char String[16];


	if ( gstResponse[EcuIdx].bDTCSMADSupported == TRUE &&
	     gstResponse[EcuIdx].DTCSMADCount != 0 )
	{
		// if not TEST 10.10 or 14 or 11.1, log ECU ID
		if ( !((geTestPhase == eTestInUseCounters       && (gTestSubsection == 10 || gTestSubsection == 14 )) ||
		       (geTestPhase == eTestPerformanceCounters &&  gTestSubsection == 1)) )
		{
			// include ECU ID in log entry
			sprintf_s ( String, sizeof ( String ),
			            "ECU %-8X  ",
			            GetEcuId ( EcuIdx ) );
		}
		else
		{
			// if Test 10.10, do not include ECU ID in log entry
			sprintf_s ( String, sizeof ( String ),
			            "" );
		}

		// log SMAD names and values
		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "%sDTC %c%02X%02X %02X  %-16s = %d\n",
		      String,
		      DTCTypeCharacter[(gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.LowByte,                                   // Failure Type Byte
		      szSmadData1,
		      gstResponse[EcuIdx].DTCSMADList[Index].NumCounts );

		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "%sDTC %c%02X%02X %02X  %-16s = %d\n",
		      String,
		      DTCTypeCharacter[(gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.LowByte,                                   // Failure Type Byte
		      szSmadData2,
		      gstResponse[EcuIdx].DTCSMADList[Index].DenomCounts );

		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "%sDTC %c%02X%02X %02X  %-16s = %0.3f\n",
		      String,
		      DTCTypeCharacter[(gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
		      gstResponse[EcuIdx].DTCSMADList[Index].Record.Dtc.LowByte,                                   // Failure Type Byte
		      szSmadData3,
		      (float)(gstResponse[EcuIdx].DTCSMADList[Index].ActRatio)/255 );
	
	}
}




/*******************************************************************************
**
**  Function:  VerifyDTCIMReadinessList
**
**  Purpose:   
**
*******************************************************************************/
STATUS VerifyDTCIMReadinessList ( void )
{
	BYTE     EcuIdx;
	REQ_MSG  stReqMsg;
	STATUS   eRetCode = PASS;

	BYTE     Index;


	//.clear previous count for all ECUs
	for ( EcuIdx = 0;
	      EcuIdx < gNumOfECUs;
	      EcuIdx++ )
	{
		gstResponse[EcuIdx].DTCIMReadinessCount = 0;
	}

	for ( Index = 0x01;
	      Index <= MAX_IM_READINESS_GROUP;
	      Index++ )
	{
		stReqMsg.SID     = 0x19;
		stReqMsg.NumIds  = 3;
		stReqMsg.u.ID[0] = 0x56;   // ReportDTCInformationByDTCReadinessGroupIdentifier
		stReqMsg.u.ID[1] = 0x33;   // FunctionalGroupIdentifier: Emission
		stReqMsg.u.ID[2] = Index;  // DTCReadinessGroupIdentifier

		// Request the DTCs which support Extended Data $xx
		if ( RequestSID ( &stReqMsg, REQ_MSG_NORMAL|REQ_MSG_IGNORE_NO_RESPONSE ) == FAIL )
		{
			Log ( FAILURE, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
			      "SID $19 $56 $33 $%02X request failed\n",
			      Index );
			eRetCode = FAIL;
		}
		else
		{
			for ( EcuIdx = 0;
			      EcuIdx < gNumOfECUs;
			      EcuIdx++ )
			{
				// Save the responses
				SaveDTCIMReadinessList ( EcuIdx );

				// Display the responses
				LogDTCIMReadinessList ( EcuIdx, Index );
			}
		}
	}

	return eRetCode;
}


/*******************************************************************************
**
**  Function:  SaveDTCIMReadinessList
**
**  Purpose:   Copy from gstResponse[EcuIdx].DTCIMRCurrent into the matching DTC
**             in gstResponse[EcuIdx].DTCSMADList.
**
*******************************************************************************/
void SaveDTCIMReadinessList ( BYTE  EcuIdx )
{
	BYTE      Index1;
	BYTE      Index2;


	// find current DTC in DTC IM Readiness list and copy the data
	for ( Index1 = 0;
	      Index1 < gstResponse[EcuIdx].DTCIMReadinessCurrentCount;
	      Index1++ )
	{
		for ( Index2 = 0;
		      Index2 < gstResponse[EcuIdx].DTCIMReadinessCount;
		      Index2++ )
		{
			if ( gstResponse[EcuIdx].DTCIMReadinessList[Index2].Record.Dtc.HighByte == gstResponse[EcuIdx].DTCIMReadinessCurrent[Index1].Record.Dtc.HighByte &&
			     gstResponse[EcuIdx].DTCIMReadinessList[Index2].Record.Dtc.MidByte  == gstResponse[EcuIdx].DTCIMReadinessCurrent[Index1].Record.Dtc.MidByte  &&
			     gstResponse[EcuIdx].DTCIMReadinessList[Index2].Record.Dtc.LowByte  == gstResponse[EcuIdx].DTCIMReadinessCurrent[Index1].Record.Dtc.LowByte  &&
			     gstResponse[EcuIdx].DTCIMReadinessList[Index2].RGID                == gstResponse[EcuIdx].DTCIMReadinessCurrent[Index1].RGID )
			{
				break;
			}
		}  // end for ( Index2 )

		// if the current DTC was not in the DTC IM Readiness list, add it
		if ( Index2 >= gstResponse[EcuIdx].DTCIMReadinessCount )
		{
			// save the data to the main IM Readiness list
			memcpy ( &gstResponse[EcuIdx].DTCIMReadinessList[Index2],
			         &gstResponse[EcuIdx].DTCIMReadinessCurrent[Index1],
			         sizeof ( DTCIMREADINESS ) );

			gstResponse[EcuIdx].DTCIMReadinessCount++;
		}
	}  // end for ( Index1 )
}


/*******************************************************************************
**
**  Function:  LogDTCIMReadinessList
**
**  Purpose:
**
*******************************************************************************/
void LogDTCIMReadinessList ( BYTE EcuIdx,
                             BYTE Index )
{
	BYTE      DTCIndex;


	if ( gstResponse[EcuIdx].bDTCIUMPRSupported == TRUE )
	{
		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "ECU %-8X  supports the following DTCs with IM Readiness Group $%02X (%s):\n",
		      GetEcuId ( EcuIdx ),
		      Index,
		      szIMReadinessGroup[Index] );

		// log IM Readiness names and values
		for ( DTCIndex = 0;
		      DTCIndex < gstResponse[EcuIdx].DTCIMReadinessCount;
		      DTCIndex++ )
		{
			if ( gstResponse[EcuIdx].DTCIMReadinessList[DTCIndex].RGID == Index )
			{
				Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
				      "              %c%02X%02X (Failure Type = $%02X  Status = $%02X)\n",
				       DTCTypeCharacter[(gstResponse[EcuIdx].DTCIMReadinessList[DTCIndex].Record.Dtc.HighByte & 0xC0) >> 6],  // 1st character (P, C, B, U)
				       gstResponse[EcuIdx].DTCIMReadinessList[DTCIndex].Record.Dtc.HighByte & 0x3F,                           // 2nd (0,1,2,3) and 3rd (0-F) characters
				       gstResponse[EcuIdx].DTCIMReadinessList[DTCIndex].Record.Dtc.MidByte,                                   // 4th and 5th characters (0-F)
				       gstResponse[EcuIdx].DTCIMReadinessList[DTCIndex].Record.Dtc.LowByte,                                   // Failure Type Byte
				       gstResponse[EcuIdx].DTCIMReadinessList[DTCIndex].Record.Status );                                      // Status of DTC
			}  // end if ( RGID == Index )
		}  // end for ( DTCIndex )
	}  // end if ( bDTCIUMPRSupported == TRUE )
	else
	{
		Log ( INFORMATION, SCREENOUTPUTON, LOGOUTPUTON, NO_PROMPT,
		      "ECU %-8X  supports no DTCs with IM Readiness Group $%02X (%s)\n",
		      GetEcuId ( EcuIdx ),
		      Index,
		      szIMReadinessGroup[Index] );
	}
}
